//	TorusGames3DMaze.c
//
//	We want the maze to be fully visible, so rather than
//	drawing traditional maze walls, we draw the maze
//	as an n×n×n lattice of nodes connected by edges.
//	
//	Integer coordinates (h,v,d) label the nodes, for h,v,d ∈ {0, 1, … , n-1}.
//	The coordinates h, v and d increase in the direction
//	of the x, y and z axes, respectively.
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesLocalization.h"
#include "GeometryGamesMatrix44.h"
#include "GeometryGamesSound.h"
#include <math.h>
#include <float.h>
#ifdef GAME_CONTENT_FOR_SCREENSHOT
#include <stdio.h>	//	for printf()
#endif


//	Connecting tube radius, as fraction of "node stride".
#define TUBE_RADIUS		0.0625

//	Slider radius, as fraction of "node stride".
#define SLIDER_RADIUS	0.1875

//	When the user lifts his/her finger/mousebutton, if the slider
//	sits within SNAP_TO_NODE_TOLERANCE of an edge's endpoint,
//	we'll snap to that node.  The edge coordinate runs from 0 to 1
//	so, for example, a tolerance of 0.05 means that
//	on the first 5% of the edge,
//		the slider would snap to the edge's base node, and
//	on the last 5% of the edge,
//		the slider would snap to the edge's terminating node.
#define SNAP_TO_NODE_TOLERANCE	0.125

//	With a mouse interface, we may insist that
//	the user click directly on the slider to move it.
//
//	With a touch interface, the user may prefer to touch slightly below the slider,
//	so s/he can still see it even with his/her finger down.
//	To allow that, we should accept touches in a larger "halo".
//
#ifdef TORUS_GAMES_2D_TOUCH_INTERFACE
#define ALLOW_HALO_HITS
#endif
#ifdef ALLOW_HALO_HITS
#define HALO_RADIUS_FACTOR	3.0
#endif

#ifdef MAKE_GAME_CHOICE_ICONS
//	Hard code a simple and aesthetically pleasing maze
//	for the screenshot and button graphic.
#define HARDCODED_SIMPLE_MAZE
#endif


typedef struct
{
	Maze3DNodePosition	itsSelf,
						itsParent;
} MazeTreeNode;


//	Public functions with private names
static void			MazeReset(ModelData *md);
static bool			MazeDragBegin(ModelData *md, HitTestRay3D *aRay);
static void			MazeDragObject(ModelData *md, HitTestRay3D *aRay, double aMotion[2]);
static void			MazeDragEnd(ModelData *md, bool aTouchSequenceWasCancelled);
static unsigned int	MazeGridSize(ModelData *md);
static void			MazeRefreshMessage(ModelData *md);

//	Private functions
static void	CreateNewMaze(ModelData *md);
static void	SetAllEdgesFalse(ModelData *md);
static void	InitPotentialEdgeList(unsigned int aMazeSize, Maze3DEdge aPotentialEdgeList[MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * 3]);
static void RandomizePotentialEdgeList(
				Maze3DEdge aPotentialEdgeList[MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * 3],
				unsigned int aPotentialEdgeListLength);
static void SetDesiredEdges(unsigned int aMazeSize,
				Maze3DNodeEdges someMazeNodes[MAX_MAZE_3D_SIZE][MAX_MAZE_3D_SIZE][MAX_MAZE_3D_SIZE],
				TopologyType aTopology,
				Maze3DEdge aPotentialEdgeList[MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * 3],
				unsigned int aPotentialEdgeListLength);
static void	GetAdjacentNode(unsigned int aMazeSize, TopologyType aTopology,
				Maze3DNodePosition aStartNode, unsigned int aDirection, bool aPositiveSense,
				Maze3DNodePosition *aFinishNode, double aCellPlacement[4][4]);
static void	PositionSliderAndGoal(ModelData *md);
static void	FindFurthestPoint(ModelData *md, Maze3DNodePosition aNearNode, Maze3DNodePosition *aFarNode);
static void	DragSliderFromNode(ModelData *md, double aMotion[2]);
static void	DragSliderAlongEdge(ModelData *md, double aMotion[2]);
static void	SnapSliderToNode(Maze3DSlider *aSlider, unsigned int aMazeSize, TopologyType aTopology, double aTolerance, double aCellPlacement[4][4]);

static void	GetPolyhedronPlacementsForOneSliceThroughNodes(
				ModelData *md, unsigned int anAxis, double anIntercept,
				double aFrameCellCenterInGameCell[4], double aRotationalPartOfSlicePlacement[4][4], double aGameCellIntoFrameCell[4][4],
				unsigned int *aCircularSlicePlacementBufferLengthPtr,		TorusGames3DPolyhedronPlacementAsCArrays **aCircularSlicePlacementBufferPtr);
static void	GetPolyhedronPlacementsForOneSliceThroughTubes(
				ModelData *md, unsigned int anAxis, double anIntercept,
				double aFrameCellCenterInGameCell[4], double aRotationalPartOfSlicePlacement[4][4], double aGameCellIntoFrameCell[4][4],
				unsigned int *aRectangularSlicePlacementBufferLengthPtr,	TorusGames3DPolyhedronPlacementAsCArrays **aRectangularSlicePlacementBufferPtr,
				unsigned int *aCircularSlicePlacementBufferLengthPtr,		TorusGames3DPolyhedronPlacementAsCArrays **aCircularSlicePlacementBufferPtr);
static void	GetPolyhedronPlacementsForOneSliceThroughSlider(
				ModelData *md, unsigned int anAxis, double anIntercept,
				double aFrameCellCenterInGameCell[4], double aRotationalPartOfSlicePlacement[4][4], double aGameCellIntoFrameCell[4][4],
				unsigned int *aCircularSlicePlacementBufferLengthPtr,		TorusGames3DPolyhedronPlacementAsCArrays **aCircularSlicePlacementBufferPtr);

static void	GetSliderPositionInGameCell(Maze3DSlider *aSliderLocation, unsigned int aMazeSize, double aSliderPosition[4]);
static void	GetGoalPositionInGameCell(Maze3DNodePosition aGoalNode, unsigned int aMazeSize, double aGoalPosition[4]);


#ifdef __APPLE__
#pragma mark -
#pragma mark set up
#endif


void Maze3DSetUp(ModelData *md)
{
	//	Initialize function pointers.
	md->itsGameShutDown					= NULL;
	md->itsGameReset					= &MazeReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= NULL;
	md->itsGame2DDragBegin				= NULL;
	md->itsGame2DDragObject				= NULL;
	md->itsGame2DDragEnd				= NULL;
	md->itsGame3DDragBegin				= &MazeDragBegin;
	md->itsGame3DDragObject				= &MazeDragObject;
	md->itsGame3DDragEnd				= &MazeDragEnd;
	md->itsGame3DGridSize				= &MazeGridSize;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= NULL;
	md->itsGameRefreshMessage			= &MazeRefreshMessage;

	//	Initialize variables.
	md->itsGameOf.Maze3D.itsSize	= 0;	//	MazeReset() will override this.
	
	//	Ask that the how-to-play instructions be shown.
	md->itsGameOf.Maze3D.itsShowInstructionsFlag = true;	//	Set this flag before calling MazeReset().

	//	Create a maze.
	MazeReset(md);
}


#ifdef __APPLE__
#pragma mark -
#pragma mark reset
#endif


static void MazeReset(ModelData *md)
{
#ifdef GAME_CONTENT_FOR_SCREENSHOT
#if 0	//	experiment with different seeds, to see which looks good?
		static unsigned int	theSeed	= 0;

		RandomInitWithSeed(theSeed);
		printf("using 3D maze seed %d\n", theSeed);
		theSeed++;
#else	//	use our favorite seed
		//	Caution:  The random number generator's implementation is different
		//	on different platforms, so while a given seed generates the same maze
		//	on iOS and macOS, we'd need to use a different seed on other platforms.
		RandomInitWithSeed(51);
#endif	//	experiment or use favorite seed

	//	Set a good-looking orientation for seed selected above.
	//
	//		Note:  To find such an orientation, enable PREPARE_FOR_SCREENSHOT
	//		which will ask MouseMove3D() to print each manually chosen orientation
	//		to the console.
	//
	md->its3DFrameCellIntoWorld = (Isometry) {0.675603, 0.038222, 0.734230, -0.054826};
#endif

	switch (md->itsDifficultyLevel)
	{
		case 0:  md->itsGameOf.Maze3D.itsSize = 2;  break;
		case 1:  md->itsGameOf.Maze3D.itsSize = 3;  break;
		case 2:  md->itsGameOf.Maze3D.itsSize = 4;  break;
		case 3:  md->itsGameOf.Maze3D.itsSize = 5;  break;
		default: md->itsGameOf.Maze3D.itsSize = 2;  break;	//	should never occur
	}
#ifdef HARDCODED_SIMPLE_MAZE
	md->itsGameOf.Maze3D.itsSize = 2;
#endif
	GEOMETRY_GAMES_ASSERT(	md->itsGameOf.Maze3D.itsSize <= MAX_MAZE_3D_SIZE,
							"Maze size exceeds maximum");

	//	Create the maze.
	CreateNewMaze(md);

	//	Place the slider and the goal as far from each other as possible.
	PositionSliderAndGoal(md);

	//	Abort any pending simulation.
	SimulationEnd(md);	//	currently unused

	//	Show the how-to-play instructions, if appropriate.
	MazeRefreshMessage(md);

	//	Ready to go.
	md->itsGameIsOver = false;
	md->itsChangeCount++;
}


static void CreateNewMaze(ModelData *md)
{
	//	This algorithm is from the Maze program written
	//	by Jeff Weeks for Adam Weeks Marano, Christmas 1992.

	//	The plan is to give each node its own index.  
	//	We then put all the possible outbound edges on a list, 
	//	randomize the order of the list,
	//	and then go down it one item at a time.
	//	Whenever a possible edge connects two nodes with different indices,
	//	add the edge to the maze and merge the neighboring indices
	//	(that is, set all occurrences of one index to equal the value
	//	of the other index).  When a potential edge connects two nodes
	//	of the same index, omit it from the maze.  It's easy to see that
	//	this will yield a maze with a unique path between any two nodes.

	unsigned int	n;
	Maze3DEdge		thePotentialEdgeList[MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * 3];

	n = md->itsGameOf.Maze3D.itsSize;

	SetAllEdgesFalse(md);
	InitPotentialEdgeList(n, thePotentialEdgeList);
	RandomizePotentialEdgeList(thePotentialEdgeList, n*n*n*3);
	SetDesiredEdges(n, md->itsGameOf.Maze3D.itsEdges, md->itsTopology, thePotentialEdgeList, n*n*n*3);
}


static void SetAllEdgesFalse(ModelData *md)
{
	unsigned int	i,
					j,
					k,
					l;

	for (i = 0; i < md->itsGameOf.Maze3D.itsSize; i++)
		for (j = 0; j < md->itsGameOf.Maze3D.itsSize; j++)
			for (k = 0; k < md->itsGameOf.Maze3D.itsSize; k++)
				for (l = 0; l < 3; l++)
				{
					md->itsGameOf.Maze3D.itsEdges[i][j][k].itsOutbound[l] = false;
					md->itsGameOf.Maze3D.itsEdges[i][j][k].itsInbound[l]  = false;
				}
}


static void InitPotentialEdgeList(
	unsigned int	aMazeSize,
	Maze3DEdge		aPotentialEdgeList[MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * 3])
{
	Maze3DEdge		theMazeEdge;
	unsigned int	theCount;

	theCount = 0;

	for (theMazeEdge.itsStartingNode.p[0] = 0; theMazeEdge.itsStartingNode.p[0] < aMazeSize; theMazeEdge.itsStartingNode.p[0]++)
		for (theMazeEdge.itsStartingNode.p[1] = 0; theMazeEdge.itsStartingNode.p[1] < aMazeSize; theMazeEdge.itsStartingNode.p[1]++)
			for (theMazeEdge.itsStartingNode.p[2] = 0; theMazeEdge.itsStartingNode.p[2] < aMazeSize; theMazeEdge.itsStartingNode.p[2]++)
				for (theMazeEdge.itsDirection = 0; theMazeEdge.itsDirection < 3; theMazeEdge.itsDirection++)
					aPotentialEdgeList[theCount++] = theMazeEdge;
}


static void RandomizePotentialEdgeList(
	Maze3DEdge		aPotentialEdgeList[MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * 3],
	unsigned int	aPotentialEdgeListLength)
{
	unsigned int	theLastIndex,
					theRandomIndex;
	Maze3DEdge		theSwapEdge;
	
	GEOMETRY_GAMES_ASSERT(	aPotentialEdgeListLength > 0,
							"Empty potential edge list");

	theLastIndex = aPotentialEdgeListLength;

	while (--theLastIndex > 0)
	{
		theRandomIndex = RandomUnsignedInteger() % (theLastIndex + 1);

		theSwapEdge							= aPotentialEdgeList[theLastIndex];
		aPotentialEdgeList[theLastIndex]	= aPotentialEdgeList[theRandomIndex];
		aPotentialEdgeList[theRandomIndex]	= theSwapEdge;
	}
}


static void SetDesiredEdges(
	unsigned int	aMazeSize,
	Maze3DNodeEdges	someMazeNodes[MAX_MAZE_3D_SIZE][MAX_MAZE_3D_SIZE][MAX_MAZE_3D_SIZE],
	TopologyType	aTopology,
	Maze3DEdge		aPotentialEdgeList[MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * MAX_MAZE_3D_SIZE * 3],
	unsigned int	aPotentialEdgeListLength)
{
	unsigned int		theCount,
						i,
						j,
						k,
						theNodeIndices[MAX_MAZE_3D_SIZE][MAX_MAZE_3D_SIZE][MAX_MAZE_3D_SIZE];
	Maze3DNodePosition	theStartNode,
						theFinishNode;
	unsigned int		theDirection,	//	0, 1 or 2
						theStartIndex,
						theFinishIndex,
						ii,
						jj,
						kk;

	//	Initialize the node indices to arbitrary but distinct values.
	theCount = 0;
	for (i = 0; i < aMazeSize; i++)
		for (j = 0; j < aMazeSize; j++)
			for (k = 0; k < aMazeSize; k++)
				theNodeIndices[i][j][k] = theCount++;

	//	Go down aPotentialEdgeList and add an edge whenever
	//	it connects two nodes of different indices.
	for (i = 0; i < aPotentialEdgeListLength; i++)
	{
		theStartNode = aPotentialEdgeList[i].itsStartingNode;
		theDirection = aPotentialEdgeList[i].itsDirection;
		GetAdjacentNode(aMazeSize, aTopology, theStartNode, theDirection, true, &theFinishNode, NULL);

		theStartIndex	= theNodeIndices[ theStartNode.p[0]][ theStartNode.p[1]][ theStartNode.p[2]];
		theFinishIndex	= theNodeIndices[theFinishNode.p[0]][theFinishNode.p[1]][theFinishNode.p[2]];

		if (theStartIndex != theFinishIndex)
		{
			//	Replace all occurrences of theFinishIndex with theStartIndex.
			for (ii = 0; ii < aMazeSize; ii++)
				for (jj = 0; jj < aMazeSize; jj++)
					for (kk = 0; kk < aMazeSize; kk++)
						if (theNodeIndices[ii][jj][kk] == theFinishIndex)
							theNodeIndices[ii][jj][kk] = theStartIndex;

			//	Add the edge to the maze.
			someMazeNodes[ theStartNode.p[0]][ theStartNode.p[1]][ theStartNode.p[2]].itsOutbound[theDirection] = true;
			someMazeNodes[theFinishNode.p[0]][theFinishNode.p[1]][theFinishNode.p[2]].itsInbound [theDirection] = true;
		}
	}

#ifdef HARDCODED_SIMPLE_MAZE
	someMazeNodes[0][0][0].itsOutbound[0] = true;
	someMazeNodes[0][0][0].itsInbound [0] = false;
	someMazeNodes[0][0][0].itsOutbound[1] = false;
	someMazeNodes[0][0][0].itsInbound [1] = false;
	someMazeNodes[0][0][0].itsOutbound[2] = false;
	someMazeNodes[0][0][0].itsInbound [2] = false;

	someMazeNodes[0][0][1].itsOutbound[0] = false;
	someMazeNodes[0][0][1].itsInbound [0] = false;
	someMazeNodes[0][0][1].itsOutbound[1] = true;
	someMazeNodes[0][0][1].itsInbound [1] = false;
	someMazeNodes[0][0][1].itsOutbound[2] = false;
	someMazeNodes[0][0][1].itsInbound [2] = false;

	someMazeNodes[0][1][0].itsOutbound[0] = false;
	someMazeNodes[0][1][0].itsInbound [0] = true;
	someMazeNodes[0][1][0].itsOutbound[1] = false;
	someMazeNodes[0][1][0].itsInbound [1] = false;
	someMazeNodes[0][1][0].itsOutbound[2] = true;
	someMazeNodes[0][1][0].itsInbound [2] = false;

	someMazeNodes[0][1][1].itsOutbound[0] = false;
	someMazeNodes[0][1][1].itsInbound [0] = false;
	someMazeNodes[0][1][1].itsOutbound[1] = false;
	someMazeNodes[0][1][1].itsInbound [1] = true;
	someMazeNodes[0][1][1].itsOutbound[2] = false;
	someMazeNodes[0][1][1].itsInbound [2] = true;

	someMazeNodes[1][0][0].itsOutbound[0] = false;
	someMazeNodes[1][0][0].itsInbound [0] = true;
	someMazeNodes[1][0][0].itsOutbound[1] = false;
	someMazeNodes[1][0][0].itsInbound [1] = true;
	someMazeNodes[1][0][0].itsOutbound[2] = true;
	someMazeNodes[1][0][0].itsInbound [2] = false;

	someMazeNodes[1][0][1].itsOutbound[0] = false;
	someMazeNodes[1][0][1].itsInbound [0] = false;
	someMazeNodes[1][0][1].itsOutbound[1] = true;
	someMazeNodes[1][0][1].itsInbound [1] = false;
	someMazeNodes[1][0][1].itsOutbound[2] = false;
	someMazeNodes[1][0][1].itsInbound [2] = true;

	someMazeNodes[1][1][0].itsOutbound[0] = true;
	someMazeNodes[1][1][0].itsInbound [0] = false;
	someMazeNodes[1][1][0].itsOutbound[1] = true;
	someMazeNodes[1][1][0].itsInbound [1] = false;
	someMazeNodes[1][1][0].itsOutbound[2] = false;
	someMazeNodes[1][1][0].itsInbound [2] = false;

	someMazeNodes[1][1][1].itsOutbound[0] = false;
	someMazeNodes[1][1][1].itsInbound [0] = false;
	someMazeNodes[1][1][1].itsOutbound[1] = false;
	someMazeNodes[1][1][1].itsInbound [1] = true;
	someMazeNodes[1][1][1].itsOutbound[2] = false;
	someMazeNodes[1][1][1].itsInbound [2] = false;
#endif
}


static void GetAdjacentNode(
	unsigned int		aMazeSize,				//	input
	TopologyType		aTopology,				//	input
	Maze3DNodePosition	aStartNode,				//	input
	unsigned int		aDirection,				//	input;  0, 1 or 2, for h, v or d
	bool				aPositiveSense,			//	input;  true for positive, false for negative
	Maze3DNodePosition	*aFinishNode,			//	output
	double				aCellPlacement[4][4])	//	input and output;  may be NULL
{
	bool			theWrapFlag	= false;
	unsigned int	theSwapValue,
					d;
	double			thePrefactor[4][4];

	//	Copy aStartNode to aFinishNode.
	*aFinishNode = aStartNode;

	//	Advance aFinishNode one step in the requested direction and sense.
	//	Wrap, if necessary, to stay within the n×n×n fundamental domain.
	if (aPositiveSense)
	{
		aFinishNode->p[aDirection]++;
		if (aFinishNode->p[aDirection] == aMazeSize)
		{
			aFinishNode->p[aDirection] = 0;
			theWrapFlag = true;
		}
	}
	else	//	negative sense
	{
		if (aFinishNode->p[aDirection] == 0)
		{
			aFinishNode->p[aDirection] = aMazeSize;
			theWrapFlag = true;
		}
		aFinishNode->p[aDirection]--;
	}
	
	//	If aFinishNode wrapped around, then...
	if (theWrapFlag)
	{
		//	...account for the topology.
		switch (aTopology)
		{
			case Topology3DTorus:
				//	Do nothing.
				break;
			
			case Topology3DKlein:
				//	If the node wrapped in the y direction,
				//	then flip its x position.
				if (aDirection == 1)
					aFinishNode->p[0] = (aMazeSize - 1) - aFinishNode->p[0];
				break;
			
			case Topology3DQuarterTurn:
				//	If the node wrapped in the z direction,
				//	then rotate its x and y coordinates by a quarter turn.
				if (aDirection == 2)
				{
					//	Realize the quarter turn 
					//	as the composition of two reflections.

					//	First reflect across the diagonal.
					theSwapValue		= aFinishNode->p[0];
					aFinishNode->p[0]	= aFinishNode->p[1];
					aFinishNode->p[1]	= theSwapValue;
					
					//	Then reflect vertically for a positive wrap
					//	or horizontally for a negative wrap.
					//
					//	Technical note:  For a positive wrap,
					//	the group rotates the fundamental cube
					//	by a 1/4 counterclockise turn when moving in the z direction.
					//	Here we're applying the inverse of that motion:
					//	the aFinishNode's z coordinate has decreased
					//	from aMazeSize to 0, and so the rotation is clockwise
					//	(when viewed by an observer looking in the positive z direction).
					//	Similar considerations apply in the case of a negative wrap.
					// 
					d = (aPositiveSense ? 1 : 0);	//	reflection direction
					aFinishNode->p[d] = (aMazeSize - 1) - aFinishNode->p[d];
				}
				break;
			
			case Topology3DHalfTurn:
				//	If the node wrapped in the z direction,
				//	then rotate its x and y coordinates by a half turn.
				if (aDirection == 2)
				{
					aFinishNode->p[0] = (aMazeSize - 1) - aFinishNode->p[0];
					aFinishNode->p[1] = (aMazeSize - 1) - aFinishNode->p[1];
				}
				break;
			
			default:
				GeometryGamesFatalError(u"Invalid topology in GetAdjacentNode() in TorusGames3DMaze.c (1)", u"Internal Error");
				break;
		}
		
		//	If the game cell has been placed in frame cell space,
		//	the caller may ask us to update its placement,
		//	so aFinishNode's frame cell position is consistent
		//	with aStartNode's.
		if (aCellPlacement != NULL)
		{
			//	In all cases, we'll need to premultiply
			//	by at least a translation.
			Matrix44Identity(thePrefactor);
			thePrefactor[3][aDirection] = (aPositiveSense ?  1.0 : -1.0);
			
			//	We may also need to account for the topology.
			switch (aTopology)
			{
				case Topology3DTorus:
					break;
					
				case Topology3DKlein:
					if (aDirection == 1)			//	If the node wrapped in the y direction,
						thePrefactor[0][0] = -1.0;	//	then reflect in the x direction.
					break;
				
				case Topology3DQuarterTurn:
					if (aDirection == 2)			//	If the node wrapped in the z direction,
					{								//	then rotate a quarter turn in the xy plane.
						thePrefactor[0][0] = 0.0;
						thePrefactor[0][1] = (aPositiveSense ?  1.0 : -1.0);
						thePrefactor[1][0] = (aPositiveSense ? -1.0 :  1.0);
						thePrefactor[1][1] = 0.0;
					}
					break;
				
				case Topology3DHalfTurn:
					if (aDirection == 2)			//	If the node wrapped in the z direction,
					{								//	then rotate a half turn in the xy plane.
						thePrefactor[0][0] = -1.0;
						thePrefactor[1][1] = -1.0;
					}
					break;
				
				default:
					GeometryGamesFatalError(u"Invalid topology in GetAdjacentNode() in TorusGames3DMaze.c (2)", u"Internal Error");
			}
			
			//	Apply thePrefactor to aCellPlacement.
			Matrix44Product(thePrefactor, aCellPlacement, aCellPlacement);
		}
	}
}


static void PositionSliderAndGoal(ModelData *md)
{
	//	Proposition
	//
	//	Let node A be an arbitrary node in the maze.
	//	Let node B be a node maximally far from node A
	//		(following the maze, of course).
	//	Let node C be a node maximally far from node B
	//		(following the maze, of course).
	//	Then the distance from node B to node C is
	//		greater than or equal to the distance between
	//		any other pair of nodes.
	//
	//	The proof is easy, but not quite so brief that
	//	I feel like writing it down here.  Presumably this
	//	result appears somewhere in the literature, even
	//	though I couldn't find it on the web (and Patti Lock
	//	was unaware of it).

	Maze3DNodePosition	theNodeA = {{0,0,0}},
						theNodeB,
						theNodeC;

	//	Find a pair of maximally distant points in the maze.
	FindFurthestPoint(md, theNodeA, &theNodeB);
	FindFurthestPoint(md, theNodeB, &theNodeC);

	//	Place the slider at theNodeB.
	md->itsGameOf.Maze3D.itsSlider.itsStatus	= SliderAtNode;
	md->itsGameOf.Maze3D.itsSlider.itsNode		= theNodeB;
	md->itsGameOf.Maze3D.itsSlider.itsDirection	= 0;	//	ignored
	md->itsGameOf.Maze3D.itsSlider.itsDistance	= 0.0;	//	ignored
	
	//	Place the goal at theNodeC.
	md->itsGameOf.Maze3D.itsGoalNode = theNodeC;

#ifdef HARDCODED_SIMPLE_MAZE
	md->itsGameOf.Maze3D.itsSlider.itsStatus	= SliderOnOutboundEdge;
	md->itsGameOf.Maze3D.itsSlider.itsNode		= (Maze3DNodePosition) {{0,0,0}};
	md->itsGameOf.Maze3D.itsSlider.itsDirection	= 0;
	md->itsGameOf.Maze3D.itsSlider.itsDistance	= 0.6;
#endif
}


static void FindFurthestPoint(
	ModelData			*md,
	Maze3DNodePosition	aNearNode,
	Maze3DNodePosition	*aFarNode)
{
	unsigned int		n;
	MazeTreeNode		*theQueue		= NULL;
	Maze3DNodePosition	theInvalidNode	= {{0xFF, 0xFF, 0xFF}};
	unsigned int		theQueueStart,
						theQueueEnd;
	Maze3DNodePosition	theSelf			= {{0x00, 0x00, 0x00}},	//	initialize to suppress compiler warning
						theParent,
						theNeighbor;
	Maze3DNodeEdges		theEdges;
	unsigned int		theDirection,
						theSense;

	n = md->itsGameOf.Maze3D.itsSize;

	theQueue = (MazeTreeNode *) GET_MEMORY(n * n * n * sizeof(MazeTreeNode));
	if (theQueue == NULL)
	{
		GeometryGamesErrorMessage(	u"Couldn't get memory to properly position slider and goal in maze.",
						u"FindFurthestPoint() Error");
		aFarNode->p[0] = (n > 1) ? ( ! aNearNode.p[0] ) : 0;
		aFarNode->p[1] = 0;
		aFarNode->p[2] = 0;
		return;
	}

	theQueueStart			= 0;
	theQueueEnd				= 0;
	theQueue[0].itsSelf		= aNearNode;
	theQueue[0].itsParent	= theInvalidNode;

	while (theQueueStart <= theQueueEnd)
	{
		//	Pull the next node off the queue.
		theSelf		= theQueue[theQueueStart].itsSelf;
		theParent	= theQueue[theQueueStart].itsParent;
		theQueueStart++;

		//	Add each of theSelf's accessible neighbors,
		//	excluding its own parent, to the queue.
		theEdges = md->itsGameOf.Maze3D.itsEdges[theSelf.p[0]][theSelf.p[1]][theSelf.p[2]];
		for (theDirection = 0; theDirection < 3; theDirection++)
		{
			for (theSense = 0; theSense < 2; theSense++)	//	0 = inbound; 1 = outbound
			{
				if (theSense ?
					theEdges.itsOutbound[theDirection] :
					theEdges.itsInbound [theDirection])
				{
					GetAdjacentNode(n,
									md->itsTopology,
									theSelf,
									theDirection,
									theSense ? true : false,
									&theNeighbor,
									NULL);

					if (theNeighbor.p[0] != theParent.p[0]
					 || theNeighbor.p[1] != theParent.p[1]
					 || theNeighbor.p[2] != theParent.p[2])
					{
						theQueueEnd++;
						theQueue[theQueueEnd].itsSelf	= theNeighbor;
						theQueue[theQueueEnd].itsParent	= theSelf;
					}
				}
			}
		}
	}
	GEOMETRY_GAMES_ASSERT(	theQueueStart == n * n * n,
							"Wrong number of elements processed on queue");

	//	We did a breadth first traversal of the tree,
	//	so the last node considered must be maximally far from the first.
	*aFarNode = theSelf;

	FREE_MEMORY_SAFELY(theQueue);
}


#ifdef __APPLE__
#pragma mark -
#pragma mark drag
#endif


static bool MazeDragBegin(
	ModelData		*md,
	HitTestRay3D	*aRay)
{
	bool		theHitFlag;
	double		theSliderRadius;
#ifdef ALLOW_HALO_HITS
	double		theHitRadius;
#endif
	signed int	m = 0,	//	initialize to suppress compiler warnings
				x,
				y,
				z;
	double		theSliderCenterInGameCell[4],
				theSliderCenterInTiling[4],
				theGameCellIntoTiling[4][4],
				theGameCellIntoFrameCell[4][4],
				theFrameCellIntoWorld[4][4],
				theSmallestT,
				theNewT;
	
	theHitFlag		= false;
	theSmallestT	= DBL_MAX;
	
	theSliderRadius	= SLIDER_RADIUS / (double)md->itsGameOf.Maze3D.itsSize;

#ifdef ALLOW_HALO_HITS
	theHitRadius = HALO_RADIUS_FACTOR * theSliderRadius;
#endif

	GetSliderPositionInGameCell(&md->itsGameOf.Maze3D.itsSlider,
								md->itsGameOf.Maze3D.itsSize,
								theSliderCenterInGameCell);

	RealizeIsometryAs4x4MatrixInSO3(&md->its3DFrameCellIntoWorld,
									theFrameCellIntoWorld);

	switch (md->itsViewType)
	{
		case ViewBasicLarge:

			//	Test for hits on the game cell's central image
			//	and its nearest neighbors -- that is, on the 3×3×3 grid
			//	of nearest images -- to guarantee that all visible images
			//	of the slider get tested.

			m = 1;	//	coordinates run -1 to +1

			break;
		
		case ViewRepeating:

			//	Test for hits over a larger grid, so the user
			//	may select images sitting deeper in the tiling.

			m = 2;	//	coordinates run -2 to +2 (anything deeper is in the fog)

			break;
		
		default:
			GeometryGamesFatalError(u"MazeDragBegin() received unexpected ViewType.", u"Internal Error");
			break;
	}

	for (x = -m; x <= +m; x++)
	{
		for (y = -m; y <= +m; y++)
		{
			for (z = -m; z <= +m; z++)
			{
				Make3DGameCellIntoTiling(theGameCellIntoTiling, x, y, z, md->itsTopology);

				Matrix44RowVectorTimesMatrix(	theSliderCenterInGameCell,
												theGameCellIntoTiling,
												theSliderCenterInTiling);
			
				if
				(
					(Ray3DIntersectsSphere(	aRay,
											theSliderCenterInTiling,
											theSliderRadius,
											&theNewT)
					 && theNewT > aRay->tMin
					 &&	theNewT < aRay->tMax
					 && theNewT < theSmallestT)
#ifdef ALLOW_HALO_HITS
				 ||
					//	Accept taps in a larger "halo" surrounding the slider.
					//	The comments accompanying the definition of ALLOW_HALO_HITS
					//	explain why this is necessary on iOS.
					//
					//	Note #1:  When the slider sits near a edge of the frame cell,
					//	the near side of the halo sphere may get cut off by one wall,
					//	while its far side gets cut off by a different wall.
					//	In such cases the user must tap the slider directly.
					//
					//	Note #2:  Testing for hits on the surface of the ball
					//	of radius theHitRadius is *not* redundant with testing
					//	for hits on the surface of the ball of radius theSliderRadius,
					//	for the reason explained in Note #1.
					//
					(Ray3DIntersectsSphere(	aRay,
											theSliderCenterInTiling,
											theHitRadius,
											&theNewT)
					 && theNewT > aRay->tMin
					 &&	theNewT < aRay->tMax
					 && theNewT < theSmallestT)
#endif
				)
				{
					theHitFlag		= true;
					theSmallestT	= theNewT;
					Matrix44Product(	theGameCellIntoTiling,
										md->its3DTilingIntoFrameCell,
										theGameCellIntoFrameCell);
					Matrix44Product(	theGameCellIntoFrameCell,
										theFrameCellIntoWorld,
										md->itsGameOf.Maze3D.itsDragCellIntoWorld);
				}
			}
		}
	}

	return theHitFlag;
}


static void MazeDragObject(
	ModelData		*md,
	HitTestRay3D	*aRay,
	double			aMotion[2])
{
	//	Responding to the relative motion (aMotion) provides
	//	a more intuitive user interface than responding
	//	to the absolute position (aRay) would.
	UNUSED_PARAMETER(aRay);

	switch (md->itsGameOf.Maze3D.itsSlider.itsStatus)
	{
		case SliderAtNode:
			DragSliderFromNode(md, aMotion);
			break;
		
		case SliderOnOutboundEdge:
			DragSliderAlongEdge(md, aMotion);
			break;
	}
}

static void DragSliderFromNode(
	ModelData	*md,
	double		aMotion[2])
{
	unsigned int		theMazeSize;
	Maze3DNodePosition	theNodePosition;
	double				(*theGameCellIntoWorld)[4];
	Maze3DNodeEdges		theEdges;
	unsigned int		i;
	double				theNodeLocationInGameCell[4],
						theNodeLocationInWorld[4],
						(*theBasisVectors)[4],
						theDerivative[4][4],
						theProjectedBasisVectors[4][4],
						theLength,
						theMotionComponent,
						theBestMotionComponent;
	unsigned int		theGameAxis,					//	0, 1 or 2 for x, y or z
						theBestAxis			= 0;		//	0, 1 or 2 for x, y or z
	bool				theBestIsOutbound	= false;	//	true for outbound edge, false for inbound edge

	//	The slider is sitting at a node.
	//	Decide which of the incident edges (if any) to move it along.
	
	//	For convenience...
	theMazeSize				= md->itsGameOf.Maze3D.itsSize;
	theNodePosition			= md->itsGameOf.Maze3D.itsSlider.itsNode;
	theGameCellIntoWorld	= md->itsGameOf.Maze3D.itsDragCellIntoWorld;
	theEdges				= md->itsGameOf.Maze3D.itsEdges[theNodePosition.p[0]][theNodePosition.p[1]][theNodePosition.p[2]];

	//	Compute the node's game cell coordinates.
	for (i = 0; i < 3; i++)
		theNodeLocationInGameCell[i] = -0.5 + (0.5 + theNodePosition.p[i])/theMazeSize;
	theNodeLocationInGameCell[3] = 1.0;

	//	Compute the node's world coordinates.
	Matrix44RowVectorTimesMatrix(	theNodeLocationInGameCell,
									theGameCellIntoWorld,
									theNodeLocationInWorld);

	//	The first three rows of theGameCellIntoWorld give
	//	the directions in world space of the standard
	//	game cell basis vectors (1,0,0), (0,1,0) and (0,0,1).
	//
	//	In practice those vectors have a final 0, like this
	//	(1,0,0,0), (0,1,0,0) and (0,0,1,0), and the matrix
	//	contains a fourth row that we'll ignore (but which
	//	must be present when we do a 4×4 matrix multiplication!).
	theBasisVectors = theGameCellIntoWorld;

	//	Still working in world space, an observer at (0,0,-1)
	//	projects the scene onto the plane z = -1/2 via the function
	//
	//		              1/2      1/2  
	//		p(x,y,z) = ( ----- x, ----- y, -1/2 )
	//		             z + 1    z + 1
	//
	//	Compute the partial derivatives telling how the projected point
	//	responds to movements of the original point (x,y,z).
	Compute3DProjectionDerivative(theNodeLocationInWorld, theDerivative);
	
	//	Apply theDerivative to project theBasisVectors onto the plane at z = -1/2.
	Matrix44Product(theBasisVectors, theDerivative, theProjectedBasisVectors);
	
	//	Normalize each of theProjectedBasisVectors to unit length, if possible.
	//	Note that only the first two components may be nonzero.
	for (theGameAxis = 0; theGameAxis < 3; theGameAxis++)
	{
		theLength = sqrt( theProjectedBasisVectors[theGameAxis][0] * theProjectedBasisVectors[theGameAxis][0]
						+ theProjectedBasisVectors[theGameAxis][1] * theProjectedBasisVectors[theGameAxis][1]);
		if (theLength > 0.0)
		{
			theProjectedBasisVectors[theGameAxis][0] /= theLength;
			theProjectedBasisVectors[theGameAxis][1] /= theLength;
		}
	}
	
	//	Which of the node's incident edges best matches theMotion?
	//	If theMotion doesn't have a positive component
	//	in the direction of any incident edge, leave 
	//	itsSlider.itsStatus as SliderAtNode and wait for the user
	//	to drag in some more appropriate direction.
	
	theBestMotionComponent = 0.0;

	for (theGameAxis = 0; theGameAxis < 3; theGameAxis++)
	{
		//	Project aMotion onto theProjectedBasisVectors[theGameAxis].
		//	The latter is a unit vector (or zero) so the dot product
		//	gives the length of aMotion's projection.
		theMotionComponent = aMotion[0] * theProjectedBasisVectors[theGameAxis][0]
					       + aMotion[1] * theProjectedBasisVectors[theGameAxis][1];

		if (theEdges.itsOutbound[theGameAxis] && +theMotionComponent > theBestMotionComponent)
		{
			theBestMotionComponent	= +theMotionComponent;
			theBestAxis				= theGameAxis;
			theBestIsOutbound		= true;
		}
		else
		if (theEdges.itsInbound[theGameAxis]  && -theMotionComponent > theBestMotionComponent)
		{
			theBestMotionComponent	= -theMotionComponent;
			theBestAxis				= theGameAxis;
			theBestIsOutbound		= false;
		}
	}

	//	Did we find an acceptable edge to move along?
	if (theBestMotionComponent > 0.0)
	{
		//	itsStatus
		md->itsGameOf.Maze3D.itsSlider.itsStatus = SliderOnOutboundEdge;

		//	itsNode
		if (theBestIsOutbound)
		{
			//	itsNode gets reinterpreted as the edge's base node,
			//	but its value remains unchanged.
		}
		else	//	theBestProjection is onto an inbound node
		{
			//	Find the inbound node's base node.
			GetAdjacentNode(theMazeSize,
							md->itsTopology,
							theNodePosition,
							theBestAxis,
							false,
							&md->itsGameOf.Maze3D.itsSlider.itsNode,
							md->itsGameOf.Maze3D.itsDragCellIntoWorld);	//	update as needed
		}
		
		//	itsDirection
		md->itsGameOf.Maze3D.itsSlider.itsDirection = theBestAxis;

		//	itsDistance
		//
		//	Set the obvious initial value, and then...
		//
		md->itsGameOf.Maze3D.itsSlider.itsDistance = (theBestIsOutbound ? 0.0 : 1.0);
		
		//	...call DragSliderAlongEdge() to account for aMotion.
		DragSliderAlongEdge(md, aMotion);
	}
}

static void DragSliderAlongEdge(
	ModelData	*md,
	double		aMotion[2])
{
	unsigned int	theMazeSize;
	unsigned int	theDirection;	//	0, 1 or 2 for x, y or z
	double			(*theGameCellIntoWorld)[4],
					theSliderPositionInGameCell[4],
					theSliderPositionInWorld[4],
					*theEdgeVectorInWorld,			//	3D unit vector
					theDerivative[4][4],			//	upper-left 3×2 block is non-zero
					theProjectedEdgeVector[4],		//	four components needed to use Matrix44f functions,
													//		only first two components are nonzero
					theProjectedLength,
					theProjectedEdgeDirection[2],	//	2D unit vector
					theMotionComponent;

	//	The slider is running along an edge.
	//	Advance it in response to aMotion,
	//	and check whether it's reached an endpoint.
	
	//	For legibility...
	theMazeSize				= md->itsGameOf.Maze3D.itsSize;
	theDirection			= md->itsGameOf.Maze3D.itsSlider.itsDirection;
	theGameCellIntoWorld	= md->itsGameOf.Maze3D.itsDragCellIntoWorld;

	//	Compute the slider's cell coordinates.
	GetSliderPositionInGameCell(&md->itsGameOf.Maze3D.itsSlider,
								theMazeSize,
								theSliderPositionInGameCell);

	//	Compute the slider's world coordinates.
	Matrix44RowVectorTimesMatrix(	theSliderPositionInGameCell,
									theGameCellIntoWorld,
									theSliderPositionInWorld);

	//	Compute the edge direction in world coordinates, as a unit vector.
	theEdgeVectorInWorld = theGameCellIntoWorld[theDirection];

	//	Compute the partial derivatives telling how motion
	//	in any direction in world space maps to the projected motion
	//	on the plane at z = -1/2, as seen by an observer at (0,0,-1).
	Compute3DProjectionDerivative(theSliderPositionInWorld, theDerivative);
	
	//	Apply theDerivative to project theEdgeVectorInWorld
	//	onto the plane at z = -1/2.
	Matrix44RowVectorTimesMatrix(	theEdgeVectorInWorld,
									theDerivative,
									theProjectedEdgeVector);

	//	How long is theProjectedEdgeVector?
	theProjectedLength = sqrt( theProjectedEdgeVector[0] * theProjectedEdgeVector[0]
							 + theProjectedEdgeVector[1] * theProjectedEdgeVector[1]);
	
	if (theProjectedLength > 0.0)
	{
		//	Compute a unit vector in the direction of theProjectedEdgeVector.
		theProjectedEdgeDirection[0] = theProjectedEdgeVector[0] / theProjectedLength;
		theProjectedEdgeDirection[1] = theProjectedEdgeVector[1] / theProjectedLength;

		//	A dot product gives the component of aMotion in theProjectedEdgeDirection.
		theMotionComponent = aMotion[0] * theProjectedEdgeDirection[0]
					       + aMotion[1] * theProjectedEdgeDirection[1];
		
		//	Scale theMotionComponent from the projection plane
		//	back to the original maze edge.  This correction accounts
		//	for the edge's distance and also for the possible foreshortening
		//	due to the fact that the edge may tilt towards or away from
		//	the observer.
		theMotionComponent /= theProjectedLength;
	}
	else	//	theProjectedLength == 0.0
	{
		//	The user is viewing the active edge end-on,
		//	and therefore cannot slide the slider along it.
		//	The user may rotate the frame cell
		//	to get a better view of the active edge.
		theMotionComponent = 0.0;
	}

	//	Convert theMotionComponent from game cell coordinates
	//	to an edge distance.
	theMotionComponent *= theMazeSize;
	
	//	Update the slider position.
	md->itsGameOf.Maze3D.itsSlider.itsDistance += theMotionComponent;

	//	If the slider has reached or passed an endpoint,
	//	snap it to that endpoint and set itsStatus to SliderAtNode.
	SnapSliderToNode(	&md->itsGameOf.Maze3D.itsSlider,
						theMazeSize,
						md->itsTopology,
						0.0,	//	snap on overshoot, but ignore undershoot
						md->itsGameOf.Maze3D.itsDragCellIntoWorld);
}


static void MazeDragEnd(
	ModelData	*md,
	bool		aTouchSequenceWasCancelled)
{
	//	Note:  If the user begins a trackpad gesture while
	//	a mouse motion is already in progress, or vice versa,
	//	the earlier motion might generate MazeDragBegin()
	//	and zero or more MazeDragObject() calls, but
	//	with no concluding MazeDragEnd().
	//	This is harmless with the current code.

	//	If a touch sequence got cancelled (perhaps because a gesture was recognized),
	//	return without snapping the slider to a node or testing for a win.
	if (aTouchSequenceWasCancelled)
		return;

	//	If the slider is close to a node, snap to it exactly.
	SnapSliderToNode(	&md->itsGameOf.Maze3D.itsSlider,
						md->itsGameOf.Maze3D.itsSize,
						md->itsTopology,
						SNAP_TO_NODE_TOLERANCE,
						md->itsGameOf.Maze3D.itsDragCellIntoWorld);
	
	//	Did the slider reach the goal?
	if ( ! md->itsGameIsOver
	 && md->itsGameOf.Maze3D.itsSlider.itsStatus == SliderAtNode
	 && md->itsGameOf.Maze3D.itsSlider.itsNode.p[0] == md->itsGameOf.Maze3D.itsGoalNode.p[0]
	 && md->itsGameOf.Maze3D.itsSlider.itsNode.p[1] == md->itsGameOf.Maze3D.itsGoalNode.p[1]
	 && md->itsGameOf.Maze3D.itsSlider.itsNode.p[2] == md->itsGameOf.Maze3D.itsGoalNode.p[2])
	{
		md->itsGameIsOver = true;

		EnqueueSoundRequest(u"MazeSolved3D.mid");

		md->itsGameOf.Maze3D.itsShowInstructionsFlag = false;	//	Ask that the how-to-play instructions not be shown.
		MazeRefreshMessage(md);
	}
}

static void SnapSliderToNode(
	Maze3DSlider	*aSlider,				//	input and output
	unsigned int	aMazeSize,				//	input
	TopologyType	aTopology,				//	input
	double			aTolerance,				//	input;  limits "undershoot" but not "overshoot"
	double			aCellPlacement[4][4])	//	input and output;  may be NULL
{
	//	If aSlider has status SliderOnOutboundEdge,
	//	but sits close to or beyond either endpoint,
	//	move the slider to that endpoint and change
	//	itsStatus to SliderAtNode.

	if (aSlider->itsStatus == SliderOnOutboundEdge)
	{
		if (aSlider->itsDistance <= 0.0 + aTolerance)
		{
			//	Reinterpret itsNode as an exact location 
			//	rather than as an edge's base node,
			//	but keep its value the same.
			aSlider->itsStatus		= SliderAtNode;
			aSlider->itsDirection	= 0;	//	ignored
			aSlider->itsDistance	= 0.0;	//	ignored
		}
		else
		if (aSlider->itsDistance >= 1.0 - aTolerance)
		{
			//	Reinterpret itsNode as an exact location 
			//	rather than as an edge's base node,
			//	and set its value to the edge's far endpoint.
			GetAdjacentNode(aMazeSize,
							aTopology,
							aSlider->itsNode,
							aSlider->itsDirection,
							true,
							&aSlider->itsNode,
							aCellPlacement);	//	update as needed
			aSlider->itsStatus		= SliderAtNode;
			aSlider->itsDirection	= 0;	//	ignored
			aSlider->itsDistance	= 0.0;	//	ignored
		}
		else
		{
			//	Leave aSlider untouched, in the interior of the edge,
			//	with status SliderOnOutboundEdge.
		}
	}
}


#ifdef __APPLE__
#pragma mark -
#pragma mark grid
#endif


static unsigned int	MazeGridSize(ModelData *md)
{
	return md->itsGameOf.Maze3D.itsSize;
}


#ifdef __APPLE__
#pragma mark -
#pragma mark status message
#endif


static void MazeRefreshMessage(ModelData *md)
{
	SetTorusGamesStatusMessage(
		md->itsGameOf.Maze3D.itsShowInstructionsFlag ?
			GetLocalizedText(u"Drag slider to goal") :
			u"",	//	NULL is OK on iOS, but not on macOS
		(ColorP3Linear) {0.0625, 0.0625, 0.0625, 1.0},
		Game3DMaze);
}


#ifdef __APPLE__
#pragma mark -
#pragma mark polyhedron placements
#endif

unsigned int GetNum3DMazePolyhedra_BufferPart1_Solids(
	ModelData	*md)
{
	unsigned int	n;
	
	n = md->itsGameOf.Maze3D.itsSize;
	
	return
		n * n * n		//	n³ nodes
	  + n * n * n - 1	//	n³ - 1 tubes (they make a "tree", so there's always one fewer tube than there are nodes)
	  + 1				//	1 slider
	  + 1;				//	1 goal
}

unsigned int GetNum3DMazePolyhedra_BufferPart2_RectangularSlices(
	ModelData	*md)
{
	unsigned int	n;
	
	n = md->itsGameOf.Maze3D.itsSize;
	
	return 3 *				//	rectangular slices; at most 3 frame cell faces will be visible,
		(					//		each frame cell face may intersect at most
			2 * n * (n + 1)	//	n·(n+1) horizontal tubes parallel to the frame cell face and
							//	n·(n+1)  vertical  tubes parallel to the frame cell face
		);
}

unsigned int GetNum3DMazePolyhedra_BufferPart3_CircularSlices(
	ModelData	*md)
{
	unsigned int	n;
	
	n = md->itsGameOf.Maze3D.itsSize;
	
	return 3 *					//	circular slices; at most 3 frame cell faces will be visible,
		(						//		each frame cell face may intersect
			(n + 1) * (n + 1)	//	(n+1)² nodes
		  + (n + 1) * (n + 1)	//	(n+1)² orthogonal tubes
		  + 1					//	1 slider
		);
}

unsigned int GetNum3DMazePolyhedra_BufferPart4_ClippedEllipticalSlices(
	ModelData	*md)
{
	return 0;	//	3D Maze needs no elliptical cross sections
}


void Get3DMazePolyhedronPlacementsForSolids(
	ModelData									*md,					//	input
	unsigned int								aPlacementBufferLength,	//	input (output buffer length in placements, not bytes)
	TorusGames3DPolyhedronPlacementAsCArrays	*aPlacementBuffer)		//	output buffer with space for aPlacementBufferLength placements
{
	unsigned int								n;
	double										theNodeStride,
												theTubeRadius,
												theTubeHalfLength;
	TorusGames3DPolyhedronPlacementAsCArrays	*thePlacement;
	unsigned int								i,
												j,
												k,
												m;
	Maze3DNodeEdges								*theNodeEdges;
	unsigned int								theNumTubes;
	bool										theNumReflectionsIsOdd;
	unsigned int								d;
	double										theTubePlacement[4][4],
												theSliderRadius,
												theGoalHalfWidth;

	GEOMETRY_GAMES_ASSERT(md->itsGame == Game3DMaze, "Game3DMaze must be active");

	n				= md->itsGameOf.Maze3D.itsSize;
	theNodeStride	= 1.0 / (double)n;

	theTubeRadius		= TUBE_RADIUS * theNodeStride;
	theTubeHalfLength	=     0.5     * theNodeStride;

	thePlacement= aPlacementBuffer;


	//	nodes

	for (i = 0; i < n; i++)
	{
		for (j = 0; j < n; j++)
		{
			for (k = 0; k < n; k++)
			{
				//	Dilational part
				for (m = 0; m < 3; m++)
					thePlacement->itsDilation[m] = theTubeRadius;

				//	Note where to find theNodeEdges.
				theNodeEdges = &md->itsGameOf.Maze3D.itsEdges[i][j][k];

				//	For itsIsometricPlacement, start with an identity matrix.
				Matrix44Identity(thePlacement->itsIsometricPlacement);

				//	If a tube goes straight through the node in some direction...
				if ((theNodeEdges->itsOutbound[0] && theNodeEdges->itsInbound[0])
				 || (theNodeEdges->itsOutbound[1] && theNodeEdges->itsInbound[1])
				 || (theNodeEdges->itsOutbound[2] && theNodeEdges->itsInbound[2]))
				{
					//	...then no sphere need be drawn at this node,
					//	and so itsIsometricPlacement may remain an identity matrix
					//	because it won't get used.
				}
				else
				{
					//	Only 1/2, 1/4 or 1/8 of the sphere will be visible
					//	according to whether 1, 2 or 3 tubes meet at this node.
					theNumTubes	= (theNodeEdges->itsOutbound[0] ? 1 : 0)
								+ (theNodeEdges->itsInbound [0] ? 1 : 0)
								+ (theNodeEdges->itsOutbound[1] ? 1 : 0)
								+ (theNodeEdges->itsInbound [1] ? 1 : 0)
								+ (theNodeEdges->itsOutbound[2] ? 1 : 0)
								+ (theNodeEdges->itsInbound [2] ? 1 : 0);

					switch (theNumTubes)
					{
						case 1:	//	1/2 sphere
							
							//	Rotate the hemisphere from the outbound side (x ≥ 0)
							//	to the inbound side (x ≤ 0)?
							if (theNodeEdges->itsOutbound[0]
							 || theNodeEdges->itsOutbound[1]
							 || theNodeEdges->itsOutbound[2])
							{
								HalfTurnY(thePlacement->itsIsometricPlacement);
							}
							
							//	Rotate the hemisphere's axis
							//	from the x axis to the y or z axis?
							if (theNodeEdges->itsOutbound[1] || theNodeEdges->itsInbound[1])
								RotateXYZ(thePlacement->itsIsometricPlacement);
							if (theNodeEdges->itsOutbound[2] || theNodeEdges->itsInbound[2])
								RotateZYX(thePlacement->itsIsometricPlacement);
							
							break;
							
						case 2:	//	1/4 sphere
							
							//	Tubes arrive in two of the three directions {x,y,z}.
							//	The default 1/4 sphere sits in the sector x ≥ 0, y ≥ 0.
							//	First rotate the sphere about the z axis as necessary
							//	(as if z were the missing direction)...
							if ((theNodeEdges->itsInbound [0] && theNodeEdges->itsOutbound[1])
							 || (theNodeEdges->itsInbound [1] && theNodeEdges->itsOutbound[2])
							 || (theNodeEdges->itsInbound [2] && theNodeEdges->itsOutbound[0]))
							{
								SwapXY(thePlacement->itsIsometricPlacement);
								ReflectAxis(thePlacement->itsIsometricPlacement, 1);
							}
							if ((theNodeEdges->itsOutbound[0] && theNodeEdges->itsInbound [1])
							 || (theNodeEdges->itsOutbound[1] && theNodeEdges->itsInbound [2])
							 || (theNodeEdges->itsOutbound[2] && theNodeEdges->itsInbound [0]))
							{
								SwapXY(thePlacement->itsIsometricPlacement);
								ReflectAxis(thePlacement->itsIsometricPlacement, 0);
							}
							if ((theNodeEdges->itsOutbound[0] && theNodeEdges->itsOutbound[1])
							 || (theNodeEdges->itsOutbound[1] && theNodeEdges->itsOutbound[2])
							 || (theNodeEdges->itsOutbound[2] && theNodeEdges->itsOutbound[0]))
							{
								ReflectAxis(thePlacement->itsIsometricPlacement, 0);
								ReflectAxis(thePlacement->itsIsometricPlacement, 1);
							}

							//	...and then rotate the result to align with the correct missing axis.
							if ( ! (theNodeEdges->itsOutbound[0] || theNodeEdges->itsInbound[0]) )
								RotateXYZ(thePlacement->itsIsometricPlacement);
							if ( ! (theNodeEdges->itsOutbound[1] || theNodeEdges->itsInbound[1]) )
								RotateZYX(thePlacement->itsIsometricPlacement);
							
							break;
							
						case 3:	//	1/8 sphere

							//	Tubes arrive in all three directions {x,y,z}.
							//	The default 1/8 sphere sits in the sector x ≥ 0, y ≥ 0, z ≥ 0.
							//	Reflect it as necessary to arrive in the correct octant,
							//	after first performing an "extra" reflection
							//	to insure that the total number of reflections is even
							//	(so that counterclockise-winding faces remain
							//	counterclockise-winding).

							theNumReflectionsIsOdd = false;
							if (theNodeEdges->itsOutbound[0])
								theNumReflectionsIsOdd = ! theNumReflectionsIsOdd;
							if (theNodeEdges->itsOutbound[1])
								theNumReflectionsIsOdd = ! theNumReflectionsIsOdd;
							if (theNodeEdges->itsOutbound[2])
								theNumReflectionsIsOdd = ! theNumReflectionsIsOdd;
							if (theNumReflectionsIsOdd)
								SwapXY(thePlacement->itsIsometricPlacement);
							
							if (theNodeEdges->itsOutbound[0])
								ReflectAxis(thePlacement->itsIsometricPlacement, 0);
							if (theNodeEdges->itsOutbound[1])
								ReflectAxis(thePlacement->itsIsometricPlacement, 1);
							if (theNodeEdges->itsOutbound[2])
								ReflectAxis(thePlacement->itsIsometricPlacement, 2);
							
							break;
							
						default:
							GeometryGamesFatalError(u"Impossible number of tubes meet at node.", u"Internal Error");
					}
				}

				thePlacement->itsIsometricPlacement[3][0] = -0.5  +  (0.5 + i) * theNodeStride;
				thePlacement->itsIsometricPlacement[3][1] = -0.5  +  (0.5 + j) * theNodeStride;
				thePlacement->itsIsometricPlacement[3][2] = -0.5  +  (0.5 + k) * theNodeStride;

				for (m = 0; m < 4; m++)
					thePlacement->itsExtraClippingCovector[m] = 0.0;	//	unused

				thePlacement++;
			}
		}
	}
	
	
	//	tubes

	for (d = 0; d < 3; d++)	//	Directions 0, 1 and 2 correspond to x, y and z axes.
	{
		//	Pre-compute the rotational part of the tube's placement in the game cell.
		//	The nested loops below will override the translational part.
		Matrix44Identity(theTubePlacement);
		switch (d)
		{
			case 0:									break;	//	align tube with x-axis
			case 1:	RotateXYZ(theTubePlacement);	break;	//	align tube with y-axis
			case 2:	RotateZYX(theTubePlacement);	break;	//	align tube with z-axis
		};
		
		for (i = 0; i < n; i++)
		{
			theTubePlacement[3][0] = -0.5  +  (0.5 + i) * theNodeStride;
			if (d == 0)
				theTubePlacement[3][0] += theTubeHalfLength;

			for (j = 0; j < n; j++)
			{
				theTubePlacement[3][1] = -0.5  +  (0.5 + j) * theNodeStride;
				if (d == 1)
					theTubePlacement[3][1] += theTubeHalfLength;

				for (k = 0; k < n; k++)
				{
					theTubePlacement[3][2] = -0.5  +  (0.5 + k) * theNodeStride;
					if (d == 2)
						theTubePlacement[3][2] += theTubeHalfLength;

					if (md->itsGameOf.Maze3D.itsEdges[i][j][k].itsOutbound[d])
					{
						//	Dilation
						thePlacement->itsDilation[0] = theTubeHalfLength;	//	longitudinal scaling factor
						thePlacement->itsDilation[1] = theTubeRadius;		//	radial scaling factor
						thePlacement->itsDilation[2] = theTubeRadius;		//	radial scaling factor

						//	Isometric placement
						Matrix44Copy(thePlacement->itsIsometricPlacement, theTubePlacement);

						//	Unused extra clipping covector
						for (m = 0; m < 4; m++)
							thePlacement->itsExtraClippingCovector[m] = 0.0;	//	unused

						thePlacement++;
					}
				}
			}
		}
	}

	//	slider
	//
	//		Chirality note:
	//		Always place the slider in the game cell in a non-reflected way,
	//		even if it has slid past the ceiling in Klein space.
	//		If the slider polyhedron weren't mirror-symmtric,
	//		we'd be in trouble, because to get it to look right we'd need
	//		to place it in the game cell with a reflection, which would
	//		run afoul of our conventions on backface culling.

	theSliderRadius = SLIDER_RADIUS * theNodeStride;
	
	thePlacement->itsDilation[0] = theSliderRadius;
	thePlacement->itsDilation[1] = theSliderRadius;
	thePlacement->itsDilation[2] = theSliderRadius;

	Matrix44Identity(thePlacement->itsIsometricPlacement);

	GetSliderPositionInGameCell(
		&md->itsGameOf.Maze3D.itsSlider,
		md->itsGameOf.Maze3D.itsSize,
		thePlacement->itsIsometricPlacement[3]);	//	overwrites translational part of matrix

	for (m = 0; m < 4; m++)
		thePlacement->itsExtraClippingCovector[m] = 0.0;	//	unused

	thePlacement++;

	//	goal

	//		Let the size of the goal be such that
	//		the slider fits snugly within the goal's window.
	theGoalHalfWidth = theSliderRadius / sqrt(GOAL_WINDOW_FRACTION*GOAL_WINDOW_FRACTION + 1.0);
	
	thePlacement->itsDilation[0] = theGoalHalfWidth;
	thePlacement->itsDilation[1] = theGoalHalfWidth;
	thePlacement->itsDilation[2] = theGoalHalfWidth;

	Matrix44Identity(thePlacement->itsIsometricPlacement);

	GetGoalPositionInGameCell(
		md->itsGameOf.Maze3D.itsGoalNode,
		md->itsGameOf.Maze3D.itsSize,
		thePlacement->itsIsometricPlacement[3]);	//	overwrites translational part of matrix

	for (m = 0; m < 4; m++)
		thePlacement->itsExtraClippingCovector[m] = 0.0;	//	unused

	thePlacement++;
	
	
	//	Error check
	GEOMETRY_GAMES_ASSERT(
		thePlacement - aPlacementBuffer == GetNum3DMazePolyhedra_BufferPart1_Solids(md),
		"Internal error:  wrong number of placements written to buffer");
}


void Get3DMazePolyhedronPlacementsForOneSlice(
	ModelData									*md,										//	input
	unsigned int								anAxis,										//	input
	double										anIntercept,								//	input
	double										aFrameCellCenterInGameCell[4],				//	input;  (x,y,z,1)
	double										aRotationalPartOfSlicePlacement[4][4],		//	input
	double										aGameCellIntoFrameCell[4][4],				//	input
	unsigned int								*aRectangularSlicePlacementBufferLengthPtr,	//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aRectangularSlicePlacementBufferPtr,		//	input and output;  increment pointer after writing each placement
	unsigned int								*aCircularSlicePlacementBufferLengthPtr,	//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aCircularSlicePlacementBufferPtr)			//	input and output;  increment pointer after writing each placement
{
	GEOMETRY_GAMES_ASSERT(md->itsGame == Game3DMaze, "Game3DMaze must be active");

	GetPolyhedronPlacementsForOneSliceThroughNodes(
		md, anAxis, anIntercept,
		aFrameCellCenterInGameCell, aRotationalPartOfSlicePlacement, aGameCellIntoFrameCell,
		aCircularSlicePlacementBufferLengthPtr,		aCircularSlicePlacementBufferPtr);

	GetPolyhedronPlacementsForOneSliceThroughTubes(
		md, anAxis, anIntercept,
		aFrameCellCenterInGameCell, aRotationalPartOfSlicePlacement, aGameCellIntoFrameCell,
		aRectangularSlicePlacementBufferLengthPtr,	aRectangularSlicePlacementBufferPtr,
		aCircularSlicePlacementBufferLengthPtr,		aCircularSlicePlacementBufferPtr);

	GetPolyhedronPlacementsForOneSliceThroughSlider(
		md, anAxis, anIntercept,
		aFrameCellCenterInGameCell, aRotationalPartOfSlicePlacement, aGameCellIntoFrameCell,
		aCircularSlicePlacementBufferLengthPtr,		aCircularSlicePlacementBufferPtr);
}

static void GetPolyhedronPlacementsForOneSliceThroughNodes(
	ModelData									*md,										//	input
	unsigned int								anAxis,										//	input
	double										anIntercept,								//	input
	double										aFrameCellCenterInGameCell[4],				//	input;  (x,y,z,1)
	double										aRotationalPartOfSlicePlacement[4][4],		//	input
	double										aGameCellIntoFrameCell[4][4],				//	input
	unsigned int								*aCircularSlicePlacementBufferLengthPtr,	//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aCircularSlicePlacementBufferPtr)			//	input and output;  increment pointer after writing each placement
{
	unsigned int								n;
	double										theNodeStride,
												theNodeRadius,
												theNodeCenterInGameCell[4];
	unsigned int								i,
												j,
												k;
	bool										theNodeMayBeCulled;
	unsigned int								m;
	double										theSlicePlaneOffset,
												theSliceRadius;
	TorusGames3DPolyhedronPlacementAsCArrays	*theSlicePlacement;

	n				= md->itsGameOf.Maze3D.itsSize;
	theNodeStride	= 1.0 / (double)n;
	theNodeRadius	= TUBE_RADIUS * theNodeStride;
	
	//	All the slice placements for this slice will share the same rotational part,
	//	but with different translational parts.

	theNodeCenterInGameCell[3] = 1.0;

	for (i = 0; i < n; i++)
	{
		theNodeCenterInGameCell[0] = -0.5 + (0.5 + i)*theNodeStride;

		for (j = 0; j < n; j++)
		{
			theNodeCenterInGameCell[1] = -0.5 + (0.5 + j)*theNodeStride;

			for (k = 0; k < n; k++)
			{
				theNodeCenterInGameCell[2] = -0.5 + (0.5 + k)*theNodeStride;

				//	If this node lies wholly outside the frame cell, skip it.
				//
				//	This simple test will cull away most nodes that don't intersect the frame cell.
				//	Occasionally a node sitting outside the frame cell
				//	but near a (1-dimensional) frame cell edge may failed to get culled here,
				//	but that's OK:  such a node will get clipped away in the GPU.
				//	The purpose of this simple test is merely to avoid processing
				//	unneeded vertices in the GPU vertex function.  In other words,
				//	this test reduces the GPU's workload but isn't needed to insure correctness.
				//
				theNodeMayBeCulled = false;
				for (m = 0; m < 3; m++)
				{
					 if (theNodeCenterInGameCell[m] + theNodeRadius < aFrameCellCenterInGameCell[m] - 0.5)
					 	theNodeMayBeCulled = true;

					 if (theNodeCenterInGameCell[m] - theNodeRadius > aFrameCellCenterInGameCell[m] + 0.5)
					 	theNodeMayBeCulled = true;
				}
				if (theNodeMayBeCulled)
					continue;

				//	How far is the slice plane from the node's center?
				theSlicePlaneOffset = anIntercept - theNodeCenterInGameCell[anAxis];

				//	If this node doesn't intersect the slice plane, skip it.
				if (theSlicePlaneOffset <= -theNodeRadius
				 || theSlicePlaneOffset >= +theNodeRadius)
				{
					continue;
				}
				
				//	Compute theSliceRadius.
				//
				//		Note:  The above intersection test guarantees
				//		that the argument of the sqrt() will be positive.
				//
				theSliceRadius = sqrt(    theNodeRadius    *    theNodeRadius
									 - theSlicePlaneOffset * theSlicePlaneOffset);

				//	Locate the next availble spot to write the slice placement to.
				if (*aCircularSlicePlacementBufferLengthPtr > 0)	//	should never fail
				{
					theSlicePlacement = *aCircularSlicePlacementBufferPtr;
					(*aCircularSlicePlacementBufferPtr)++;
					(*aCircularSlicePlacementBufferLengthPtr)--;
				}
				else
				{
					//	In principle we've ensured that enough buffer space will always be available.
					//	But in case it's not...
#ifdef DEBUG
					//	...during development we'd want to stop the app and resolve the problem, but...
					GEOMETRY_GAMES_ABORT("internal error:  insufficient slice placement buffer space for maze node slice");
					theSlicePlacement = NULL;
#else
					//	...once the app is in the user's hands, we'd simply not draw this particular slice.
					continue;
#endif
				}
				
				//	Set the dilational part of the slice placement.
				//
				//		Note:  The dilational part gets applied in the slice's own local coordinate system.
				//		gCircularSliceVertices places the disk in the xy plane,
				//		with its normal vector pointing in the negative z direction,
				//		so the dilations apply to the x and y coordinate, not the z coordinate.
				//
				theSlicePlacement->itsDilation[0] = theSliceRadius;
				theSlicePlacement->itsDilation[1] = theSliceRadius;
				theSlicePlacement->itsDilation[2] = 1.0;	//	safe but ultimately irrelevant,
															//		because the disk sits at z = 0
				
				//	Compute the isometric part of the slice placement in game cell coordinates
				//	(we'll convert it to frame cell coordinates immediately below).
				Matrix44Copy(theSlicePlacement->itsIsometricPlacement, aRotationalPartOfSlicePlacement);
				for (m = 0; m < 3; m++)
					theSlicePlacement->itsIsometricPlacement[3][m] = theNodeCenterInGameCell[m];	//	value for m = anAxis gets overridden immediately below
				theSlicePlacement->itsIsometricPlacement[3][anAxis] = anIntercept;
				
				//	Convert the isometric placement from game cell to frame cell coordinates.
				Matrix44Product(	theSlicePlacement->itsIsometricPlacement,
									aGameCellIntoFrameCell,
									theSlicePlacement->itsIsometricPlacement);
				
				//	We won't need the extra clipping vector, so set it to zero.
				for (m = 0; m < 4; m++)
					theSlicePlacement->itsExtraClippingCovector[m] = 0.0;	//	unused in 3D Maze
			}
		}
	}
}

static void GetPolyhedronPlacementsForOneSliceThroughTubes(
	ModelData									*md,										//	input
	unsigned int								anAxis,										//	input
	double										anIntercept,								//	input
	double										aFrameCellCenterInGameCell[4],				//	input;  (x,y,z,1)
	double										aRotationalPartOfSlicePlacement[4][4],		//	input
	double										aGameCellIntoFrameCell[4][4],				//	input
	unsigned int								*aRectangularSlicePlacementBufferLengthPtr,	//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aRectangularSlicePlacementBufferPtr,		//	input and output;  increment pointer after writing each placement
	unsigned int								*aCircularSlicePlacementBufferLengthPtr,	//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aCircularSlicePlacementBufferPtr)			//	input and output;  increment pointer after writing each placement
{
	unsigned int								n;
	double										theNodeStride,
												theTubeRadius,
												theTubeHalfLength,
												theTubeCenterInGameCell[4],
												theBoundingBoxHalfSize[3];
	unsigned int								d,
												i,
												j,
												k;
	bool										theTubeMayBeCulled;
	unsigned int								m;
	double										theSlicePlaneOffset;
	TorusGames3DPolyhedronPlacementAsCArrays	*theSlicePlacement;
	double										theRectangleHalfWidth;


	n					= md->itsGameOf.Maze3D.itsSize;
	theNodeStride		= 1.0 / (double)n;
	theTubeRadius		= TUBE_RADIUS * theNodeStride;
	theTubeHalfLength	= 0.5 * theNodeStride;

	for (d = 0; d < 3; d++)	//	Directions 0, 1 and 2 correspond to x, y and z axes.
	{
		for (i = 0; i < n; i++)
		{
			theTubeCenterInGameCell[0] = -0.5  +  (0.5 + i) * theNodeStride;
			if (d == 0)
			{
				theTubeCenterInGameCell[0] += theTubeHalfLength;
				theBoundingBoxHalfSize[0]	= theTubeHalfLength;
			}
			else
			{
				theBoundingBoxHalfSize[0]	= theTubeRadius;
			}

			for (j = 0; j < n; j++)
			{
				theTubeCenterInGameCell[1] = -0.5  +  (0.5 + j) * theNodeStride;
				if (d == 1)
				{
					theTubeCenterInGameCell[1] += theTubeHalfLength;
					theBoundingBoxHalfSize[1]	= theTubeHalfLength;
				}
				else
				{
					theBoundingBoxHalfSize[1]	= theTubeRadius;
				}

				for (k = 0; k < n; k++)
				{
					theTubeCenterInGameCell[2] = -0.5  +  (0.5 + k) * theNodeStride;
					if (d == 2)
					{
						theTubeCenterInGameCell[2] += theTubeHalfLength;
						theBoundingBoxHalfSize[2]	= theTubeHalfLength;
					}
					else
					{
						theBoundingBoxHalfSize[2]	= theTubeRadius;
					}

					//	If no outbound maze tube is present in this location, skip it.
					if ( ! md->itsGameOf.Maze3D.itsEdges[i][j][k].itsOutbound[d] )
						continue;

					//	If this tube lies wholly outside the frame cell, skip it.
					//
					//	This simple test will cull away most tubes that don't intersect the frame cell.
					//	Occasionally a tube sitting outside the frame cell
					//	but near a (1-dimensional) frame cell edge may failed to get culled here,
					//	but that's OK:  such a tube will get clipped away in the GPU.
					//	The purpose of this simple test is merely to avoid processing
					//	unneeded vertices in the GPU vertex function.  In other words,
					//	this test reduces the GPU's workload but isn't needed to insure correctness.
					//
					theTubeMayBeCulled = false;
					for (m = 0; m < 3; m++)
					{
						 if (theTubeCenterInGameCell[m] + theBoundingBoxHalfSize[m] <= aFrameCellCenterInGameCell[m] - 0.5)
							theTubeMayBeCulled = true;

						 if (theTubeCenterInGameCell[m] - theBoundingBoxHalfSize[m] >= aFrameCellCenterInGameCell[m] + 0.5)
							theTubeMayBeCulled = true;
					}
					if (theTubeMayBeCulled)
						continue;

					//	How far is the slice plane from the tube's center?
					theSlicePlaneOffset = anIntercept - theTubeCenterInGameCell[anAxis];

					//	If this tube doesn't intersect the slice plane, skip it.
					if (theSlicePlaneOffset <= -theBoundingBoxHalfSize[anAxis]
					 || theSlicePlaneOffset >= +theBoundingBoxHalfSize[anAxis])
					{
						continue;
					}

					//	If the cross section is a rectangle, how wide is it?
					if (d != anAxis)
					{
						//	The above code guarantees that
						//
						//		|theSlicePlaneOffset| < |theTubeRadius|
						//
						//	If anything ever goes wrong, then...
						if (fabs(theSlicePlaneOffset) >= fabs(theTubeRadius))
						{
#ifdef DEBUG
							//	...during development we'd want to stop the app and resolve the problem, but...
							GEOMETRY_GAMES_ABORT("Internal error:  invalidid longitudinal slice");
#else
							//	...once the app is in the user's hands, we'd simply not draw this particular slice.
							continue;
#endif
						}
						
						//	We just checked that the argument of the sqrt() must be positive.
						theRectangleHalfWidth = sqrt(	 theTubeRadius    *    theTubeRadius
													- theSlicePlaneOffset * theSlicePlaneOffset);
					}
					else
					{
						theRectangleHalfWidth = 1.0;	//	suppress compiler warning
					}

					//	Locate the next availble spot on the appropriate buffer
					//	to write the slice placement to.
					if (d == anAxis)	//	tube is orthogonal to slice plane
					{
						if (*aCircularSlicePlacementBufferLengthPtr > 0)	//	should never fail
						{
							theSlicePlacement = *aCircularSlicePlacementBufferPtr;
							(*aCircularSlicePlacementBufferPtr)++;
							(*aCircularSlicePlacementBufferLengthPtr)--;
						}
						else
						{
							theSlicePlacement = NULL;
						}
					}
					else				//	tube is parallel to slice plane
					{
						if (*aRectangularSlicePlacementBufferLengthPtr > 0)	//	should never fail
						{
							theSlicePlacement = *aRectangularSlicePlacementBufferPtr;
							(*aRectangularSlicePlacementBufferPtr)++;
							(*aRectangularSlicePlacementBufferLengthPtr)--;
						}
						else
						{
							theSlicePlacement = NULL;
						}
					}
					//	In principle we've ensured that enough buffer space will always be available.
					//	But in case it's not...
#ifdef DEBUG
					//	...during development we'd want to stop the app and resolve the problem, but...
					GEOMETRY_GAMES_ASSERT(
						theSlicePlacement != NULL,
						"internal error:  insufficient slice placement buffer space for tube slice");
#else
					//	...once the app is in the user's hands, we'd simply not draw this particular slice.
					if (theSlicePlacement == NULL)
						continue;
#endif
			
					//	Set the dilational part of the slice placement.
					//
					//		Note:  The dilational part gets applied in the slice's own local coordinate system.
					//		gSquareSliceVertices and gCircularSliceVertices place the square or disk in the xy plane,
					//		with its normal vector pointing in the negative z direction,
					//		so the dilations apply to the x and y coordinate, not the z coordinate.
					//
					if (d == anAxis)
					{
						//	The tube is orthogonal to slice plane,
						//	so its cross section is a disk.
						theSlicePlacement->itsDilation[0] = theTubeRadius;
						theSlicePlacement->itsDilation[1] = theTubeRadius;
						theSlicePlacement->itsDilation[2] = 1.0;	//	safe but unused
					}
					else
					{
						//	The tube is parallel to slice plane,
						//	so its cross section is a rectangle.

						if (aRotationalPartOfSlicePlacement[0][d] != 0)
						{
							//	The square's x-axis maps to the tube's long direction.
							theSlicePlacement->itsDilation[0] = theTubeHalfLength;
							theSlicePlacement->itsDilation[1] = theRectangleHalfWidth;
							theSlicePlacement->itsDilation[2] = 1.0;	//	safe but unused
						}
						else
						{
							//	The square's y-axis maps to the tube's long direction.
							theSlicePlacement->itsDilation[0] = theRectangleHalfWidth;
							theSlicePlacement->itsDilation[1] = theTubeHalfLength;
							theSlicePlacement->itsDilation[2] = 1.0;	//	safe but unused
						}
					}
			
					//	Compute the isometric part of the slice placement in game cell coordinates
					//	(we'll convert it to frame cell coordinates immediately below).
					Matrix44Copy(theSlicePlacement->itsIsometricPlacement, aRotationalPartOfSlicePlacement);
					for (m = 0; m < 3; m++)
						theSlicePlacement->itsIsometricPlacement[3][m] = theTubeCenterInGameCell[m];	//	value for m = anAxis gets overridden immediately below
					theSlicePlacement->itsIsometricPlacement[3][anAxis] = anIntercept;
					
					//	Convert the isometric placement from game cell to frame cell coordinates.
					Matrix44Product(	theSlicePlacement->itsIsometricPlacement,
										aGameCellIntoFrameCell,
										theSlicePlacement->itsIsometricPlacement);
					
					//	We won't need the extra clipping vector, so set it to zero.
					for (m = 0; m < 4; m++)
						theSlicePlacement->itsExtraClippingCovector[m] = 0.0;	//	unused in 3D Maze
				}
			}
		}
	}
}

static void GetPolyhedronPlacementsForOneSliceThroughSlider(
	ModelData									*md,										//	input
	unsigned int								anAxis,										//	input
	double										anIntercept,								//	input
	double										aFrameCellCenterInGameCell[4],				//	input;  (x,y,z,1)
	double										aRotationalPartOfSlicePlacement[4][4],		//	input
	double										aGameCellIntoFrameCell[4][4],				//	input
	unsigned int								*aCircularSlicePlacementBufferLengthPtr,	//	input and output;  decrement  count  after writing each placement
	TorusGames3DPolyhedronPlacementAsCArrays	**aCircularSlicePlacementBufferPtr)			//	input and output;  increment pointer after writing each placement
{
	unsigned int								n;
	double										theNodeStride,
												theSliderRadius,
												theSliderCenterInGameCell[4];
	bool										theSliderMayBeCulled;
	unsigned int								i;
	double										theSlicePlaneOffset,
												theSliceRadius;
	TorusGames3DPolyhedronPlacementAsCArrays	*theSlicePlacement;

	n				= md->itsGameOf.Maze3D.itsSize;
	theNodeStride	= 1.0 / (double)n;
	theSliderRadius = SLIDER_RADIUS * theNodeStride;

	GetSliderPositionInGameCell(&md->itsGameOf.Maze3D.itsSlider,
								n,
								theSliderCenterInGameCell);

	//	If the slider lies wholly outside the frame cell, skip it.
	//
	//	This simple test will cull away most slider images that don't intersect the frame cell.
	//	Occasionally a slider image sitting outside the frame cell
	//	but near a (1-dimensional) frame cell edge may failed to get culled here,
	//	but that's OK:  such a slider image will get clipped away in the GPU.
	//	The purpose of this simple test is merely to avoid processing
	//	unneeded vertices in the GPU vertex function.  In other words,
	//	this test reduces the GPU's workload but isn't needed to insure correctness.
	//
	theSliderMayBeCulled = false;
	for (i = 0; i < 3; i++)
	{
		 if (theSliderCenterInGameCell[i] + theSliderRadius <= aFrameCellCenterInGameCell[i] - 0.5)
			theSliderMayBeCulled = true;

		 if (theSliderCenterInGameCell[i] - theSliderRadius >= aFrameCellCenterInGameCell[i] + 0.5)
			theSliderMayBeCulled = true;
	}
	if (theSliderMayBeCulled)
		return;

	//	How far is the slice plane from the slider's center?
	theSlicePlaneOffset = anIntercept - theSliderCenterInGameCell[anAxis];

	//	If this slider image doesn't intersect the slice plane, skip it.
	if (theSlicePlaneOffset <= -theSliderRadius
	 || theSlicePlaneOffset >= +theSliderRadius)
	{
		return;
	}
	
	//	Compute theSliceRadius.
	//
	//		Note:  The above intersection test guarantees
	//		that the argument of the sqrt() will be positive.
	//
	theSliceRadius = sqrt(   theSliderRadius   *   theSliderRadius
						 - theSlicePlaneOffset * theSlicePlaneOffset);

	//	Locate the next availble spot to write the slice placement to.
	if (*aCircularSlicePlacementBufferLengthPtr > 0)	//	should never fail
	{
		theSlicePlacement = *aCircularSlicePlacementBufferPtr;
		(*aCircularSlicePlacementBufferPtr)++;
		(*aCircularSlicePlacementBufferLengthPtr)--;
	}
	else
	{
		//	In principle we've ensured that enough buffer space will always be available.
		//	But in case it's not...
#ifdef DEBUG
		//	...during development we'd want to stop the app and resolve the problem, but...
		GEOMETRY_GAMES_ABORT("internal error:  insufficient slice placement buffer space for maze slider slice");
		theSlicePlacement = NULL;
#else
		//	...once the app is in the user's hands, we'd simply not draw this particular slice.
		return;
#endif
	}

	//	Set the dilational part of the slice placement.
	//
	//		Note:  The dilational part gets applied in the slice's own local coordinate system.
	//		gCircularSliceVertices places the disk in the xy plane,
	//		with its normal vector pointing in the negative z direction,
	//		so the dilations apply to the x and y coordinate, not the z coordinate.
	//
	theSlicePlacement->itsDilation[0] = theSliceRadius;
	theSlicePlacement->itsDilation[1] = theSliceRadius;
	theSlicePlacement->itsDilation[2] = 1.0;	//	safe but ultimately irrelevant,
												//		because the disk sits at z = 0

	//	Compute the isometric part of the slice placement in game cell coordinates
	//	(we'll convert it to frame cell coordinates immediately below).
	Matrix44Copy(theSlicePlacement->itsIsometricPlacement, aRotationalPartOfSlicePlacement);
	for (i = 0; i < 3; i++)
		theSlicePlacement->itsIsometricPlacement[3][i] = theSliderCenterInGameCell[i];	//	value for i = anAxis gets overridden immediately below
	theSlicePlacement->itsIsometricPlacement[3][anAxis] = anIntercept;
	
	//	Convert the isometric placement from game cell to frame cell coordinates.
	Matrix44Product(	theSlicePlacement->itsIsometricPlacement,
						aGameCellIntoFrameCell,
						theSlicePlacement->itsIsometricPlacement);
	
	//	We won't need the extra clipping vector, so set it to zero.
	for (i = 0; i < 3; i++)
		theSlicePlacement->itsExtraClippingCovector[i] = 0.0;	//	unused in 3D Maze
}


void Get3DMazeAxisAlignedBoundingBox(
	ModelData	*md,									//	input
	double		someBoundingBoxCornersInGameCell[2][4])	//	output;  any pair of diametrically opposite corners
{
	unsigned int	n;
	double			theNodeStride;
	Maze3DSlider	*theSlider;
	Byte			theSliderDirection;
	double			theSliderRadius,
					theSliderCenterInGameCell[4],
					theSliderFarthestPoint;

	GEOMETRY_GAMES_ASSERT(md->itsGame == Game3DMaze, "Game3DMaze must be active");

	n				= md->itsGameOf.Maze3D.itsSize;
	theNodeStride	= 1.0 / (double)n;
	
	//	Get3DMazePolyhedronPlacementsForSolids() encodes each node's "outbound" edges
	//	(the ones that go in the positive coordinate direction), not its "inbound" edges.
	//	The bounding box may (and almost always will) extend an extra half-edge-length
	//	beyond the frame cell wall in each of the positive coordinate directions,
	//	but not in the negative directions.

	someBoundingBoxCornersInGameCell[0][0] = -0.5;
	someBoundingBoxCornersInGameCell[0][1] = -0.5;
	someBoundingBoxCornersInGameCell[0][2] = -0.5;
	someBoundingBoxCornersInGameCell[0][3] =  1.0;

	someBoundingBoxCornersInGameCell[1][0] = +0.5  +  0.5 * theNodeStride;
	someBoundingBoxCornersInGameCell[1][1] = +0.5  +  0.5 * theNodeStride;
	someBoundingBoxCornersInGameCell[1][2] = +0.5  +  0.5 * theNodeStride;
	someBoundingBoxCornersInGameCell[1][3] =  1.0;

	//	Allow for the slider, which may extend slightly beyond the end of an outbound edge.
	theSlider = &md->itsGameOf.Maze3D.itsSlider;
	if (theSlider->itsStatus == SliderOnOutboundEdge)	//	slider on an edge?
	{
		theSliderDirection = theSlider->itsDirection;
		
		if (theSlider->itsNode.p[theSliderDirection] == n - 1)	//	edge extends beyond frame cell?
		{
			theSliderRadius = SLIDER_RADIUS * theNodeStride;
			GetSliderPositionInGameCell(theSlider, n, theSliderCenterInGameCell);
			
			theSliderFarthestPoint = theSliderCenterInGameCell[theSliderDirection] + theSliderRadius;
			
			if (someBoundingBoxCornersInGameCell[1][theSliderDirection] < theSliderFarthestPoint)
				someBoundingBoxCornersInGameCell[1][theSliderDirection] = theSliderFarthestPoint;
		}
	}
}

#ifdef __APPLE__
#pragma mark -
#pragma mark utilities
#endif


static void GetSliderPositionInGameCell(
	Maze3DSlider	*aSliderLocation,	//	input
	unsigned int	aMazeSize,			//	input
	double			aSliderPosition[4])	//	output, nominally [-0.5, +0.5] game cell coordinates,
										//		but slider may go a half an edge length
										//		beyond the cell wall
{
	double			thePosition[3];	//	0…(n-1) lattice coordinates, but with fractional values allowed
	unsigned int	i;

	for (i = 0; i < 3; i++)
		thePosition[i] = (double) aSliderLocation->itsNode.p[i];

	switch (aSliderLocation->itsStatus)
	{
		case SliderAtNode:
			break;

		case SliderOnOutboundEdge:
			thePosition[aSliderLocation->itsDirection] += aSliderLocation->itsDistance;
			break;
	}

	for (i = 0; i < 3; i++)
		aSliderPosition[i] = -0.5 + (0.5 + thePosition[i])/aMazeSize;

	aSliderPosition[3] = 1.0;
}

static void GetGoalPositionInGameCell(
	Maze3DNodePosition	aGoalNode,			//	input,  in 0…(n-1) lattice coordinates
	unsigned int		aMazeSize,			//	input
	double				aGoalPosition[4])	//	output, in [-0.5, +0.5] game cell coordinates
{
	unsigned int	i;

	for (i = 0; i < 3; i++)
		aGoalPosition[i] = -0.5 + (0.5 + (double)aGoalNode.p[i])/(double)aMazeSize;
	
	aGoalPosition[3] = 1.0;
}
